
import Image from "next/image";
import Link from "next/link";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { HeroSection } from "@/components/home/hero-section";
import { NewsletterForm } from "@/components/shared/newsletter-form";
import { DynamicContentSuggestions } from "@/components/shared/dynamic-content-suggestions";
import { CORE_SERVICES, APP_NAME } from "@/constants";
import { ArrowRight, CheckCircle } from "lucide-react";
import { ProjectCard } from "@/components/projects/project-card";
import { getPublicProjects } from "@/lib/actions"; // Import action to fetch projects
import type { Project } from "@/types";

export default async function HomePage() {
  const latestProjects: Project[] = await getPublicProjects(3); // Fetch 3 latest projects

  return (
    <>
      <HeroSection />

      {/* Services Overview Section */}
      <section className="py-16 lg:py-24 bg-background">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl lg:text-4xl font-bold text-foreground">Our Core Services</h2>
            <p className="mt-4 text-lg text-muted-foreground max-w-2xl mx-auto">
              Driving impact through specialized expertise in enterprise development, financial inclusion, digital innovation, and experiential marketing.
            </p>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {CORE_SERVICES.map((service) => (
              <Card key={service.id} className="flex flex-col shadow-md hover:shadow-lg transition-shadow duration-300">
                <CardHeader className="items-center text-center">
                  {service.icon && <service.icon className="h-12 w-12 text-primary mb-4" />}
                  <CardTitle className="text-xl text-primary text-shadow-primary">{service.title}</CardTitle>
                </CardHeader>
                <CardContent className="flex-grow text-center">
                  <p className="text-sm text-muted-foreground mb-4">{service.description}</p>
                  <Button variant="link" asChild className="text-primary">
                    <Link href={`/services#${service.id}`}>
                      Learn More <ArrowRight className="ml-2 h-4 w-4" />
                    </Link>
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Impact Highlights / Key Projects Teaser */}
      <section className="py-16 lg:py-24 bg-muted/30">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl lg:text-4xl font-bold text-foreground">Making a Difference</h2>
            <p className="mt-4 text-lg text-muted-foreground max-w-2xl mx-auto">
              Discover how {APP_NAME} creates lasting impact through collaborative projects and innovative solutions.
            </p>
          </div>
          {latestProjects.length > 0 ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {latestProjects.map((project) => (
                <ProjectCard key={project.id} project={project} />
              ))}
            </div>
          ) : (
            <p className="text-center text-muted-foreground">No projects to display at the moment. Check back soon!</p>
          )}
          <div className="text-center mt-12">
            <Button asChild size="lg">
              <Link href="/projects">
                View All Projects <ArrowRight className="ml-2 h-5 w-5" />
              </Link>
            </Button>
          </div>
        </div>
      </section>

      {/* Dynamic Content Suggestions Section */}
      <section className="py-16 lg:py-24 bg-background">
        <div className="container mx-auto px-4">
           <div className="text-center mb-12">
            <h2 className="text-3xl lg:text-4xl font-bold text-foreground">Want to Know More?</h2>
            <p className="mt-4 text-lg text-muted-foreground max-w-2xl mx-auto">
              Here are some questions you can ask us directly on WhatsApp.
            </p>
          </div>
          <div className="max-w-3xl mx-auto">
            <DynamicContentSuggestions />
          </div>
        </div>
      </section>

      {/* Newsletter Subscription Section */}
      <section className="py-16 lg:py-24 bg-primary/10">
        <div className="container mx-auto px-4">
          <div className="max-w-xl mx-auto text-center">
            <h2 className="text-3xl lg:text-4xl font-bold text-primary mb-4 text-shadow-primary">Stay Updated</h2>
            <p className="text-lg text-muted-foreground mb-8">
              Subscribe to our newsletter for the latest insights, news, and project updates from {APP_NAME}.
            </p>
            <NewsletterForm />
          </div>
        </div>
      </section>
    </>
  );
}
