
import type { MetadataRoute } from 'next'
import { NAV_LINKS, CORE_SERVICES } from '@/constants'; // KEY_PROJECTS removed as it will be dynamic
import { getPublicProjects } from '@/lib/actions'; // Action to fetch projects

export default async function sitemap(): Promise<MetadataRoute.Sitemap> { // Made async
  const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000';

  const staticRoutes = NAV_LINKS.map(link => ({
    url: `${baseUrl}${link.href}`,
    lastModified: new Date(),
    changeFrequency: 'monthly' as 'monthly',
    priority: link.href === '/' ? 1 : (link.href === '/intellicash' ? 0.9 : 0.8),
  }));

  const projects = await getPublicProjects(); // Fetch projects from DB
  const projectRoutes = projects.map(project => ({
    url: `${baseUrl}/projects/${project.id}`,
    lastModified: project.createdAt ? new Date(project.createdAt) : new Date(), // Use createdAt if available
    changeFrequency: 'yearly' as 'yearly',
    priority: 0.7,
  }));

  const serviceRoutes = CORE_SERVICES.map(service => ({
    url: `${baseUrl}/services/${service.id}`,
    lastModified: new Date(),
    changeFrequency: 'yearly' as 'yearly',
    priority: 0.7,
  }));

  // Staff portal routes - generally lower priority and might be excluded or marked noindex/nofollow via robots.txt or page metadata
  const staffPortalRoutes: MetadataRoute.Sitemap = [
    {
      url: `${baseUrl}/staff-portal`, // This page redirects
      lastModified: new Date(),
      changeFrequency: 'yearly',
      priority: 0.1,
    },
    {
      url: `${baseUrl}/staff-portal/login`,
      lastModified: new Date(),
      changeFrequency: 'yearly',
      priority: 0.1,
    },
    {
      url: `${baseUrl}/staff-portal/signup`,
      lastModified: new Date(),
      changeFrequency: 'yearly',
      priority: 0.1,
    },
    { // Staff dashboard main page
      url: `${baseUrl}/staff-portal/dashboard`,
      lastModified: new Date(),
      changeFrequency: 'monthly',
      priority: 0.2,
    },
    { // Admin projects list page
      url: `${baseUrl}/staff-portal/dashboard/admin-projects`,
      lastModified: new Date(),
      changeFrequency: 'monthly',
      priority: 0.2,
    },
    { // Admin images list page
      url: `${baseUrl}/staff-portal/dashboard/admin-images`,
      lastModified: new Date(),
      changeFrequency: 'monthly',
      priority: 0.2,
    },
     { // Site Settings page
      url: `${baseUrl}/staff-portal/dashboard/settings`,
      lastModified: new Date(),
      changeFrequency: 'monthly',
      priority: 0.2,
    },
    // { // Placeholder for new project page (now a modal, but route might be used later)
    //   url: `${baseUrl}/staff-portal/dashboard/admin-projects/new`, // Though it's a modal now, keeping for completeness if direct link ever used
    //   lastModified: new Date(),
    //   changeFrequency: 'yearly',
    //   priority: 0.1,
    // },
    { // User profile page
      url: `${baseUrl}/staff-portal/dashboard/profile`,
      lastModified: new Date(),
      changeFrequency: 'monthly',
      priority: 0.2,
    }
  ];


  return [
    ...staticRoutes,
    ...projectRoutes,
    ...serviceRoutes,
    ...staffPortalRoutes,
  ];
}
