
"use client";

import Link from "next/link";
import Image from "next/image";
import { usePathname } from "next/navigation";
import { useState } from "react";
import type { LucideIcon } from "lucide-react";
import { Menu, X } from "lucide-react";
import { NAV_LINKS, APP_NAME, LOGO_URL as FALLBACK_LOGO_URL } from "@/constants"; // Renamed for clarity
import { Button } from "@/components/ui/button";
import { Sheet, SheetContent, SheetTrigger, SheetClose, SheetTitle } from "@/components/ui/sheet";
import { cn } from "@/lib/utils";
import { ThemeToggle } from "@/components/layout/theme-toggle";
import { Separator } from "@/components/ui/separator";
import { useSiteSettingsContext } from "@/context/site-settings-provider"; // Import context hook

export function Navbar() {
  const pathname = usePathname();
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const siteSettings = useSiteSettingsContext();

  const currentLogoUrl = siteSettings?.logoUrl || FALLBACK_LOGO_URL;

  const renderIcon = (icon: LucideIcon | string | undefined, altText: string) => {
    if (typeof icon === 'string') {
      return <Image src={icon} alt={`${altText} icon`} width={20} height={20} className="mr-2 object-contain" />;
    }
    if (icon) { // Check if icon is a LucideIcon component
      const IconComponent = icon as LucideIcon;
      return <IconComponent className="mr-2 h-4 w-4" />;
    }
    return null;
  };
  
  const renderMobileIcon = (icon: LucideIcon | string | undefined, altText: string) => {
    if (typeof icon === 'string') {
      return <Image src={icon} alt={`${altText} icon`} width={20} height={20} className="mr-2 object-contain" />;
    }
    if (icon) { // Check if icon is a LucideIcon component
      const IconComponent = icon as LucideIcon;
      return <IconComponent className="mr-2 h-5 w-5" />;
    }
    return null;
  };


  return (
    <header className="sticky top-0 z-50 w-full border-b border-border/40 bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
      <div className="container flex h-16 max-w-screen-2xl items-center">
        <Link href="/" className="ml-2 flex items-center gap-2 mr-6" onClick={() => setIsMobileMenuOpen(false)}>
          <Image src={currentLogoUrl} alt={APP_NAME + " Logo"} width={36} height={36} className="object-contain" />
          <span className="text-xl font-bold text-primary text-shadow-primary">{APP_NAME}</span>
        </Link>

        {/* Desktop Navigation & Theme Toggle */}
        <nav className="hidden md:flex flex-1 items-center justify-start gap-1">
          {NAV_LINKS.map((link) => (
            <Button
              key={link.href}
              variant="ghost"
              asChild
              className={cn(
                "text-sm font-medium transition-colors hover:text-primary hover:bg-primary/5 px-3 py-2 h-auto",
                pathname === link.href ? "text-primary bg-primary/10" : "text-muted-foreground"
              )}
            >
              <Link href={link.href}>
                {renderIcon(link.icon, link.label)}
                {link.label}
              </Link>
            </Button>
          ))}
        </nav>
        
        <div className="flex items-center justify-end ml-auto">
          <div className="hidden md:block">
            <ThemeToggle />
          </div>

          {/* Mobile Navigation Trigger */}
          <div className="md:hidden ml-2">
            <Sheet open={isMobileMenuOpen} onOpenChange={setIsMobileMenuOpen}>
              <SheetTrigger asChild>
                <Button variant="ghost" size="icon">
                  <Menu className="h-6 w-6" />
                  <span className="sr-only">Open menu</span>
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="w-[280px] p-0 flex flex-col">
                <div className="flex items-center justify-between p-4 border-b">
                   <Link href="/" className="flex items-center gap-2" onClick={() => setIsMobileMenuOpen(false)}>
                    <Image src={currentLogoUrl} alt={APP_NAME + " Logo"} width={32} height={32} className="object-contain" />
                    <SheetTitle asChild>
                      <span className="text-lg font-bold text-primary text-shadow-primary">{APP_NAME}</span>
                    </SheetTitle>
                  </Link>
                  <SheetClose asChild>
                     <Button variant="ghost" size="icon">
                        <X className="h-6 w-6" />
                        <span className="sr-only">Close menu</span>
                      </Button>
                  </SheetClose>
                </div>
                <nav className="flex-grow flex flex-col gap-2 p-4">
                  {NAV_LINKS.map((link) => (
                    <SheetClose asChild key={link.href}>
                      <Link
                        href={link.href}
                        className={cn(
                          "text-base font-medium transition-colors hover:text-primary py-2.5 px-3 rounded-md flex items-center",
                          pathname === link.href
                            ? "bg-primary/10 text-primary"
                            : "text-foreground hover:bg-muted"
                        )}
                        onClick={() => setIsMobileMenuOpen(false)}
                      >
                        {renderMobileIcon(link.icon, link.label)}
                        {link.label}
                      </Link>
                    </SheetClose>
                  ))}
                </nav>
                <Separator className="my-2" />
                <div className="p-4 mt-auto border-t">
                  <div className="flex justify-between items-center">
                     <p className="text-sm text-muted-foreground">Switch Theme</p>
                     <ThemeToggle />
                  </div>
                </div>
              </SheetContent>
            </Sheet>
          </div>
        </div>
      </div>
    </header>
  );
}
